<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\Package;
use App\Models\Purchase;
use App\Models\Rebate;
use App\Models\User;
use App\Models\SpinChance;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class PurchaseController extends Controller
{
    public function purchaseConfirmation($id)
    {
        try {
            $package = Package::find($id);
            $user = Auth::user();
            $rebate = Rebate::first();

            if (!$package) {
                return back()->with('error', "Package not found.");
            }

            if ($package->status != 'active') {
                return back()->with('error', "Package is Inactive.");
            }

            $maxPurchases = $package->max_purchase_limit;
            $userPurchaseCount = Purchase::where('user_id', $user->id)
                ->where('package_id', $package->id)
                ->count();

            if ($userPurchaseCount >= $maxPurchases) {
                return back()->with('error', "You have reached the maximum purchase limit of $maxPurchases for this package.");
            }

            if (!$this->checkIfCanBuyPackage($user, $package)) {
                return back()->with('error', "You must purchase the required packages first.");
            }

            if ($package->price <= $user->balance) {
                User::where('id', $user->id)->update([
                    'balance' => $user->balance - $package->price,
                    'investor' => $package->price > 0 ? 1 : $user->investor
                ]);

                $purchase = new Purchase();
                $purchase->user_id = $user->id;
                $purchase->package_id = $package->id;
                $purchase->amount = $package->price;
                $purchase->daily_income = $package->commission_with_avg_amount / $package->validity;
                $purchase->hourly = ($package->commission_with_avg_amount / $package->validity) / 24;
                $purchase->date = now()->addHours(24);
                $purchase->validity = Carbon::now()->addDays($package->validity);
                $purchase->status = 'active';
                $purchase->save();

                // ✅ Upgrade VIP based on total investment
                $totalInvestment = Purchase::where('user_id', $user->id)->sum('amount');

                $vipLevel = 0;
                if ($totalInvestment >= 256000000) {
                $vipLevel = 10;
                } elseif ($totalInvestment >= 128000000) {
                $vipLevel = 9;
                } elseif ($totalInvestment >= 64000000) {
                $vipLevel = 8;
                } elseif ($totalInvestment >= 3200000) {
                $vipLevel = 7;
                } elseif ($totalInvestment >= 150000) {
                $vipLevel = 6;
                } elseif ($totalInvestment >= 78000) {
                $vipLevel = 5;
                } elseif ($totalInvestment >= 35000) {
                $vipLevel = 4;
                } elseif ($totalInvestment >= 10000) {
                $vipLevel = 3;
                } elseif ($totalInvestment >= 2000) {
                $vipLevel = 2;
                } elseif ($totalInvestment >= 500) {
                $vipLevel = 1;
                } elseif ($totalInvestment >= 100) {
                $vipLevel = 0;
                }



                if ($vipLevel > $user->vip_level) {
                    $user->vip_level = $vipLevel;
                    $user->save();
                }

                SpinChance::updateOrCreate(
                    ['user_id' => $user->id],
                    ['chances' => \DB::raw('chances + 1')]
                );

                $this->distributeCommission($user, $package, $rebate);

                return back()->with('success', "Investment Successful.");
            } else {
                return back()->with('error', "You currently have low balance.");
            }

        } catch (\Illuminate\Database\QueryException $e) {
            if ($e->getCode() == '45000') {
                return back()->with('error', "You have reached the maximum purchase limit for this package.");
            }
            return back()->with('error', "An unexpected error occurred. Please try again later.");
        }
    }

    protected function checkIfCanBuyPackage($user, $package)
    {
        $purchasedPackages = Purchase::where('user_id', $user->id)->pluck('package_id')->toArray();

        switch ($package->category) {
            case 'welfare':
                $fixed = Package::where('category', 'fixed')->first();
                if ($fixed && !in_array($fixed->id, $purchasedPackages)) {
                    return false;
                }
                break;
            case 'activity':
                $fixed = Package::where('category', 'fixed')->first();
                $welfare = Package::where('category', 'welfare')->first();
                if (
                    ($fixed && !in_array($fixed->id, $purchasedPackages)) ||
                    ($welfare && !in_array($welfare->id, $purchasedPackages))
                ) {
                    return false;
                }
                break;
        }
        return true;
    }

    protected function distributeCommission($user, $package, $rebate)
    {
        $first = User::where('ref_id', $user->ref_by)->first();
        if ($first) {
            $amount = $package->price * $rebate->team_commission1 / 100;
            User::where('id', $first->id)->increment('balance', $amount);

            $this->createLedger($first->id, $user->id, 'First Level Commission', 'first', $amount);
            $this->secondLevelCommission($first, $package, $rebate, $user);
        }
    }

    protected function secondLevelCommission($first, $package, $rebate, $user)
    {
        $second = User::where('ref_id', $first->ref_by)->first();
        if ($second) {
            $amount = $package->price * $rebate->team_commission2 / 100;
            User::where('id', $second->id)->increment('balance', $amount);

            $this->createLedger($second->id, $user->id, 'Second Level Commission', 'second', $amount);
            $this->thirdLevelCommission($second, $package, $rebate, $user);
        }
    }

    protected function thirdLevelCommission($second, $package, $rebate, $user)
    {
        $third = User::where('ref_id', $second->ref_by)->first();
        if ($third) {
            $amount = $package->price * $rebate->team_commission3 / 100;
            User::where('id', $third->id)->increment('balance', $amount);

            $this->createLedger($third->id, $user->id, 'Third Level Commission', 'third', $amount);
        }
    }

    protected function createLedger($userId, $fromUserId, $text, $step, $amount)
    {
        $ledger = new UserLedger();
        $ledger->user_id = $userId;
        $ledger->get_balance_from_user_id = $fromUserId;
        $ledger->reason = 'commission';
        $ledger->perticulation = $text;
        $ledger->amount = $amount;
        $ledger->debit = $amount;
        $ledger->status = 'approved';
        $ledger->step = $step;
        $ledger->date = now()->format('d-m-Y H:i');
        $ledger->save();
    }

    public function vip_confirm($vip_id)
    {
        $vip = Package::find($vip_id);
        return view('app.main.vip_confirm', compact('vip'));
    }

    protected function ref_user($ref_by)
    {
        return User::where('ref_id', $ref_by)->first();
    }
}
